﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.Xml;


namespace CS_iSMS_Receive_Multiple_API_Polling_Mode
{
    class iSMSparseXML
    {
        // This function will determine which protocol was used to send recvmsg, 
        // and call the appropriate parser function to return the requested message.
        public iSMSmessage GetMessage(String iSMSresponse, int MsgNum)
        {
            // Check if iSMSresponse is in the XML format
            if (iSMSresponse.Contains("<?xml"))
                return GetHTTPMessage(iSMSresponse, MsgNum); // XML format, use HTTP parser.
            else
                return GetTCPMessage(iSMSresponse, MsgNum); // Non XML, use TCP parser.
        }

        // The iSMS responds with a XML formated message when using HTTP.
        // This function parses the iSMS response to a HTTP recvmsg, and returns the request message info.
        // There can be multiple messages in response, MsgNum determines which one is parsed.
        public iSMSmessage GetHTTPMessage(String iSMSresponse, int MsgNum)
        {   
            // Create new iSMSmessage class to hold message info.
            iSMSmessage parsedSMS = new iSMSmessage();

            // Create a XmlDocument and load the iSMS response string into it.
            XmlDocument doc = new XmlDocument();
            doc.LoadXml(iSMSresponse);

            // Create a list of parent nodes of the Messages. 
            // There should be a entry from each SMS message received.
            XmlNodeList MessageList = doc.GetElementsByTagName("MessageNotification");

            // Cycle through each Message and find the Message_Index that matches the sent MsgNum
            for (int i = 0; i < MessageList.Count; ++i)
            {   // Create a list of child nodes for this Message. Each node is a element with info about the SMS (Message_Index, SendNumber, Date, Time, EncodingFlag, Message).
                XmlNodeList MsgInfoList = MessageList[i].ChildNodes;
                int MessageIndex = -1; // Flag to indicate if MsgNum message was found.

                //Cycle through each element of the message to find the Message_Index.
                for (int j = 0; j < MsgInfoList.Count; ++j)
                {   // Check if this child is the Message_Index
                    if (MsgInfoList[j].Name == "Message_Index")
                    {   // Convert Message_Index text to a number.
                        int msgID = Convert.ToInt32(MsgInfoList[j].InnerText.ToString(), 10);
                        if (msgID == MsgNum)
                        {   // MsgNum was found. Set flag and break loop.
                            MessageIndex = i;
                            break;
                        }
                    }
                }
                // If MsgNum was found in i, then set parsedSMS with this message's info.
                if (MessageIndex == i)
                {
                    parsedSMS.MessageIndex = i+1; // A successful parse is indicated by setting the Message Index value. 0=failed to find message.
                    // Cycle through message elements and fill in message info as found
                    for (int j = 0; j < MsgInfoList.Count; ++j) 
                    {
                        if (MsgInfoList[j].Name == "SenderNumber")
                            parsedSMS.SenderNumber = MsgInfoList[j].InnerText.ToString();
                        if (MsgInfoList[j].Name == "Date")
                            parsedSMS.Date = MsgInfoList[j].InnerText.ToString();
                        if (MsgInfoList[j].Name == "Time")
                            parsedSMS.Time = MsgInfoList[j].InnerText.ToString();
                        if (MsgInfoList[j].Name == "EncodingFlag")
                            parsedSMS.EncodingFlag = MsgInfoList[j].InnerText.ToString();
                        if (MsgInfoList[j].Name == "Message")
                            parsedSMS.Message = MsgInfoList[j].InnerText.ToString();
                    }
                }
            }
            // Return iSMSmessage class with SMS message values parsed.
            return parsedSMS; 
        }

        // The iSMS response is different when using TCP. It's not in a XML format.
        // This function parses the iSMS response to a TCP recvmsg, and returns the request message info.
        // There can be multiple messages in response, MsgNum determines which one is parsed.
        public iSMSmessage GetTCPMessage(String iSMSresponse, int MsgNum)
        {
            iSMSmessage parsedSMS = new iSMSmessage();

            // Find Message Index for the requested message (MsgNum)
            String Msg_index = "Msg_index_" + MsgNum;
            if (iSMSresponse.Contains(Msg_index))
            {   // Message Found

                // A successful parse is indicated by setting the Message Index value. 0=failed to find message.
                parsedSMS.MessageIndex = MsgNum;

                // Find where Message Index is in iSMS response string.
                // Each line ends with a carraige return and line feed. 
                // To find end of Message Index line and the start of the next line, seach for next line feed and add one.
                int startParse = iSMSresponse.IndexOf(Msg_index) + 12;
                int NextLineStart = iSMSresponse.IndexOfAny("\n".ToCharArray(), startParse) + 1;

                // The GetNextLine function will return the text of the line starting a NextLineStart.
                // Use GetNextLine to get SenderNumber
                String NextLineText = GetNextLine(iSMSresponse, NextLineStart);
                parsedSMS.SenderNumber = NextLineText;
                NextLineStart = NextLineStart + NextLineText.Length + 2; // Calculate start of next line. 
                // Use GetNextLine to get Date
                NextLineText = GetNextLine(iSMSresponse, NextLineStart);
                parsedSMS.Date = NextLineText;
                NextLineStart = NextLineStart + NextLineText.Length + 2;
                // Use GetNextLine to get Time
                NextLineText = GetNextLine(iSMSresponse, NextLineStart);
                parsedSMS.Time = NextLineText;
                NextLineStart = NextLineStart + NextLineText.Length + 2;
                // Use GetNextLine to get EncodingFlag
                NextLineText = GetNextLine(iSMSresponse, NextLineStart);
                parsedSMS.EncodingFlag = NextLineText;
                NextLineStart = NextLineStart + NextLineText.Length + 2;
                // Use GetNextLine to get text of the SMS Message
                NextLineText = GetNextLine(iSMSresponse, NextLineStart);
                parsedSMS.Message = NextLineText;
            }
            // Return iSMSmessage class with values parsed.
            return parsedSMS; 
        }

        // The GetNextLine function will return the text of the line starting at NextLineStart.
        // Each line ends with a carraige return and line feed. It searches for Carraige return to determine end of the line.
        private String GetNextLine(String Message, int NextLineStart)
        {
            try
            {
                int NextLineEnd = Message.IndexOfAny("\r".ToCharArray(), NextLineStart); // Find end of line (carraige return).
                String NextLineText = Message.Substring(NextLineStart, NextLineEnd - NextLineStart); // Use substring to get text from start of line to end.
                return NextLineText; // return line text.
            }
            catch (Exception)
            {
                return ""; // If error, return a blank line.
            }
        }



        // This function will determine which protocol was used to send recvmsg, 
        // and call the appropriate parser function to return Message count (number of messages sent by iSMS).
        public int GetMessageCount(String iSMSresponse)
        {
            // Check if iSMSresponse is in the XML format
            if (iSMSresponse.Contains("<?xml"))
                return GetHTTPMessageCount(iSMSresponse); // XML format, use HTTP parser.
            else
                return GetTCPMessageCount(iSMSresponse); // Non XML, use TCP parser.
        }

        public int GetHTTPMessageCount(String iSMSresponse)
        {
            // Create a XmlDocument and load the iSMS response string into it.
            XmlDocument doc = new XmlDocument();
            doc.LoadXml(iSMSresponse);

            // Create a list of parent nodes with Msg_Count name. Should only be one. 
            XmlNodeList MessageList = doc.GetElementsByTagName("Msg_Count");
            // If Msg_Count element was found, then return it's value as an int.
            if (MessageList.Count > 0)
            {
                String mCount = MessageList[0].InnerText.ToString();
                try
                {
                    return Convert.ToInt32(mCount, 10);
                }
                catch (Exception)
                {
                    return 0; // Error occured while parsing. Return 0.
                }
            }
            return 0; // Msg_Count wasn't found. Return 0
        }


        public int GetTCPMessageCount(String iSMSresponse)
        {
            
            // Make sure this is a valid iSMS message response.
            if (iSMSresponse.Contains("From: MultiModem iSMS"))
            {
                // Find a known position in message.
                // The Msg_Count should be 3 lines down from this position.
                int startParse = iSMSresponse.IndexOf("From: MultiModem iSMS");
                
                // Each line ends with a carraige return and line feed.
                // To find end of current line and the start of the next line, seach for next line feed.
                int NextLineStart = iSMSresponse.IndexOfAny("\n".ToCharArray(), startParse) + 1;
                // Skip Packet_endflag
                NextLineStart = iSMSresponse.IndexOfAny("\n".ToCharArray(), NextLineStart) + 1;
                // Skip Unread_available
                NextLineStart = iSMSresponse.IndexOfAny("\n".ToCharArray(), NextLineStart) + 1;
                // Get Msg_Count                                
                String NextLineText = GetNextLine(iSMSresponse, NextLineStart);
                try
                {
                    return Convert.ToInt32(NextLineText, 10); // Return Msg_Count as int.
                }
                catch (Exception)
                {
                    return 0;
                }
            }
            return 0;
        }
    }
 }
